/*
 * vjemm.c
 *
 * By Ross Ridge
 * Public Domain
 *
 * Emulate Origin's JEMM memory manger.
 */

#include "myvmm.h"
#include "flags.h"
#include "vjemmapi.h"
#include "vernums.h"
#include "ems.h"

#ifdef SCCS_IDS
static char const ITEXT vjemm_sccs_id[] = "@(#) vjemm vjemm.c 1.5 03/08/28 16:54:37";
#endif

/*
 * Amount of free EMS/XMS memory to report to the game as being available.
 * The game will try to allocate all free memory so this number should
 * be only as big as the biggest ammount of memory any game needs.
 */
#define MEMORY_SIZE			3*1024*1024

/*
 * Maximum number of EMS handles any game can allocate at one time.
 */
#define EMS_MAX_HANDLES			2

/*
 * Fundamental constant of IA-32 processors.
 */
#define PAGE_SIZE	4096

/*
 * Flag bits that need to be set in all page table entries (PTEs).
 */
#define PAGE_FLAGS	7

/*
 * The part of PTE that is the base address.
 */
#define PAGE_BASE_MASK	((DWORD) 0xFFFFF000)

/*
 * Number of EMS pages in a standard EMS page frame.
 */ 
#define EMS_STANDARD_FRAME_PAGES	4

/*
 * Size of an EMS page.
 */
#define EMS_PAGE_SIZE			(16*1024)

/*
 * Start, end, and length of the JEMM EMS page frame in CPU pages.
 */
#define EMS_JEMM_FRAME_BASE		0xa0
#define EMS_JEMM_FRAME_END		0x100
#define EMS_JEMM_FRAME_PAGES		(EMS_JEMM_FRAME_END 	\
					 - EMS_JEMM_FRAME_BASE)


/*
 * Amount of free EMS memory to report in EMS pages.
 */
#define EMS_MAX_STDPAGES		(MEMORY_SIZE / EMS_PAGE_SIZE)
	
/*
 * Scale factor to convert EMS pages to CPU pages.
 */				
#define EMS_PAGE_CONVERT		(EMS_PAGE_SIZE / PAGE_SIZE)

/*
 * DMA emulation constants.
 */
#define DMA_CHANNELS		4
#define DMA_MAGIC_ADDRESS	0x00800000
#define DMA_MAGIC_MASK		((DWORD) 0xFFC00000)

/*
 * Linear address of the DOS High Memory Area (HMA).
 */
#define HMA_ADDRESS		0x00100000
#define HMA_PAGE		(HMA_ADDRESS / PAGE_SIZE)

/*
 * Linear address to map the VGA frame buffer in the JEMM page table
 * so that's always accessable.
 */
#define JEMM_VIDMEM_FIXED	0x00200000
#define JEMM_VIDMEM_FIXED_PAGE	(JEMM_VIDMEM_FIXED / PAGE_SIZE)

/*
 * Fake linear address to report with the JEMM 'FH' API.
 */
#define JEMM_FH_BASE		0x04000000

/*
 * Number of vectors in the IDT used by Windows.
 */
#define IDT_VECTORS	0x60

/*
 * How our interrupt handlers should the current interrupt:
 *
 *  FATAL: a fatal error occurred, enter VMM with a simulated double fault
 *  DONT_CHAIN: return directly to the game
 *  CHAIN: pass the interrupt to VMM (Windows) for normal handling.
 *  other: real mode address of callback function to push on the real mode
 *         stack, and then chain to the VMM 
 */
static DWORD const FATAL = (DWORD) -2;
static DWORD const DONT_CHAIN = (DWORD) -1;
static DWORD const CHAIN = 0;

/*
 * Saved state of a mapped EMS page.  What handle it came from and 
 * what page was it from the handle.
 */
struct ems_mapping {
	WORD handle;
	WORD page;
};

/*
 * State of an EMS handle.  The count of allocated CPU pages, the
 * CPU pages allocated, and the state of the last saved standard
 * page frame mapping.  The first CPU page allocated is used to store
 * the PTEs for the rest of the allocated CPU pages.
 */

struct ems_handle {
	DWORD count;
	DWORD *pages;
	struct ems_mapping saved_map[EMS_STANDARD_FRAME_PAGES];
};

/*
 * DMA emulation state for a channel.
 */
struct dma_data {
	BYTE addr[3];
	BYTE len[2];
	BYTE mode;
};

/*
 * Per VM data.
 */
struct vm_data {
	DWORD vm;		/* VM handle */
	char locked_v86;	/* locked the  V86 arena */
	char forced_jemm_pt;	/* JEMM enabled */
	char using_jemm_pt;	/* currently using JEMM page table */
	char using_my_stack;
	char assigned_dma_page; /* allocated DMA emulation buffer */
	char sblive_fix;	/* early EOI fix for SBLive's SB16 emu */
	BYTE standard_frame_page; /* CPU page of the standard EMS frame */
	BYTE dma_flipflop;	/* DMA emulation */
	BYTE dma_buffer_page;	/* CPU page of the DMA emulation buffer */
	BYTE dma_buffer_index;	/* current page for DMA emulation */
	DWORD *jemm_pt;		/* JEMM page table */
	DWORD jemm_pt_phys;	/* Physical address JEMM page table */ 
	DWORD save_v86_pde;	/* Physical address of standard page table */
	DWORD restore_cbfn;	/* Real mode callback function in frontend */
	BYTE *using_jemm_pt_ptr; /* Using JEMM page table flag in frontend */
	DWORD saved_dma_page;	/* Saved pages of DMA emulation buffer */
	DWORD *mu_pages;	/* pages from handle used with 'MU' API */
	DWORD *fh_pages;	/* pages from handle used with 'FH' API */
	DWORD dma_ems_handle;	/* EMS handle used with DMA emulation */
	struct dma_data dma[DMA_CHANNELS];
	struct ems_handle handles[EMS_MAX_HANDLES];
	struct ems_mapping standard_page_map[EMS_STANDARD_FRAME_PAGES];
	struct ems_mapping jemm_page_map[EMS_JEMM_FRAME_PAGES 
					/ EMS_PAGE_CONVERT];
	DWORD my_stack[2048];
};

/*
 * Format of structure used with 32-bit IA-32 LIDT and SIDT instructions.
 */
struct idtptr {
	WORD limit;
	DWORD offset;
} PACKED;

/*
 * Format of a 32-bit IA-32 IDT entry.
 */
struct idt_entry {
	WORD offset_low;
	WORD selector;
	BYTE unused;
	BYTE flags;
	WORD offset_hi;
} PACKED;

/*
 * IDT vectors in idt.asm
 */
extern DWORD const my_idt_vectors[];

/*
 * NMI handler in idt.asm
 */
extern void nmi_handler(void);
extern CBFN old_nmi_handler;
extern DWORD nmi_counter;
extern DWORD nmi_counter2;

/*
 * Forward declarations.
 */
BYTE ems_deallocate_pages(struct vm_data *vmdata, DWORD *fatal, WORD handle);
int vm_allocate(DWORD vm);
int switch_to_v86_pt(struct vm_data *vmdata);
void lognum(char const *s, DWORD n);
int unforce_jemm_pt(struct vm_data *vmdata);

Declare_DDB(VJEMM);

/*
 * Offset in the cb of each VM of the pointer to our per VM data.
 */
DWORD LDATA cb_offset = 0;

/*
 * All the allocated per VM data.
 */
HLIST LDATA vmdata_list = NULL;

/*
 * Number of VMs with allocated per VM data.  
 */
int LDATA vm_allocate_count = 0;

/*
 * Outside VMM means VMM hasn't been entered and it's not safe
 * to call VxD services.  Some simple VxD services are presumed to
 * be safe anyways.
 */
int LDATA outside_vmm = 0;

int LDATA hooked_task_switch = 0;
#ifdef USE_LOGRAM
DWORD LDATA logram_event = NULL;
#endif

/*
 * Used to test what pages are being used.
 */
HMEM break_page;
DWORD break_page_phys;

/*
 * Overwrote VMM's IDT with our own.
 */
int installed_my_idt;

/*
 * VMM's original IDT.
 */
struct idt_entry saved_vmm_idt[IDT_VECTORS];

/*
 * IDT entries whose type was changed.  Used to restore protected mode IDTs.
 * 
 */
unsigned char changed_idt_type[IDT_VECTORS];

/*
 * Invalidate the CPU's TLB.  Called after any page table modificaitons.
 */
static inline void LTEXT
invalidate_tlb(void) {
	DWORD tmp;
	asm volatile("movl %%cr3,%0; movl %0,%%cr3"
		     : "=r" (tmp));
}

#if 1
inline void * LTEXT
memset(void *d, int c, unsigned int n) {
	char *ret = d;
	asm volatile("cld; rep stosb"
		     : "+D" (d), "+c" (n)
		     : "a" ((char) c)
		     : "memory");
	return ret;
}
#endif

static inline void * LTEXT
dword_memset(DWORD *d, DWORD c, unsigned int n) {
	DWORD *ret = d;
	asm volatile("cld; rep stosl"
		     : "+D" (d), "+c" (n)
		     : "a" (c)
		     : "memory");
	return ret;
}


/*
 * Convert real mode segment:offset address to a linear address.
 */
static inline void *
segoff_to_linear(WORD seg, WORD off) {
	return (void *) ((DWORD) seg * 16 + off);
}

/*
 * Convert real mode far pointer to a linear address.
 */

static inline void *
farptr_to_linear(DWORD ptr) {
	return segoff_to_linear(ptr >> 16, (WORD) ptr);
}

/*
 * Make a real mode far pointer from a segment:offset pair
 */
static inline DWORD
make_farptr(WORD seg, WORD off) {
	return ((DWORD) seg << 16) | off;
}


/*
 * Get the Client_Reg_Struc from a VM handle.
 */
static inline struct Client_Reg_Struc *
get_crs(DWORD vm) {
	struct cb_s *cb = (struct cb_s *) vm;
	return cb->CB_Client_Pointer;
}

/*
 * Get the correct stack frame structure for the type of interrupt.
 */
static inline struct iret_frame *
get_iret_frame(struct frame *frame) {
	if (!(frame->fault & 0x80)) {
		return &frame->nocode;
	}
	return (struct iret_frame *) ((DWORD *) &frame->nocode + 1);
}

#if 1
extern int abs(int n); // Use the GCC builtin
#else
inline int LTEXT
abs(int n) {
	if (n < 0) {
		return -n;
	}
	return n;
}
#endif

extern int strlen(char const *s); // Use the GCC builtin

char * LTEXT
strncpy(char *d, char const *s, unsigned int n) {
	char *ret = d;
 
	while(n > 0) {
		char c;

		n--;
		*d++ = c = *s++;
		if (c == '\0') {
			break;
		}
	}

	while(n > 0) {
		n--;
		*d++ = '\0';
	}

	return ret;
}

#ifdef USE_LOG

#include "logvxd.c"

void LTEXT
lognote(char const *note) {
	struct lognote *ent = (struct lognote *) log_entry(LOG_NOTE);
	strncpy(ent->note, note, sizeof(ent->note));
}

void LTEXT
lognum(char const *note, DWORD num) {
	struct lognum *ent = (struct lognum *) log_entry(LOG_NUM);
	strncpy(ent->note, note, sizeof(ent->note));
	ent->num = num;
}

void LTEXT
logpte(DWORD num, DWORD pte) {
	struct logpte *ent = (struct logpte *) log_entry(LOG_PTE);
	ent->num = num;
	ent->pte = pte;
}

void LTEXT
logclient(char const *note, struct Client_Reg_Struc *crs, int flag) {
	struct logclient *ent = (struct logclient *) log_entry(LOG_CLIENT);
	strncpy(ent->note, note, sizeof(ent->note));
	ent->cs = crs->Client_CS;
	ent->ss = crs->Client_SS;
	ent->eip = crs->Client_EIP;
	ent->esp = crs->Client_ESP;
	// ent->eflags = crs->Client_EFLAGS;
	ent->flag = flag;
}

void LTEXT
logframe(char const *note, struct frame *frame, int flag) {
	DWORD *p = (DWORD *) frame;
	if (frame->fault & 0x80) {
		p++;
	}
	logclient(note, (struct Client_Reg_Struc *) p,  flag);
	// lognum("fault", frame->fault & 0xFF);
}

void LTEXT
logmap(WORD physical, WORD logical) {
	struct logmap *ent = (struct logmap *) log_entry(LOG_MAP);
	ent->physical = physical;
	ent->logical = logical;
}

void LTEXT
logregs(struct vm_data *vmdata, struct frame *frame) {
	struct iret_frame *iret = get_iret_frame(frame);
	BYTE *p;
	struct logregs *ent = (struct logregs *) log_entry(LOG_REGS);
	if (iret->eflags & FLAG_VM) {
		memcpy(&ent->frame, frame, sizeof ent->frame);
		p = segoff_to_linear(iret->pm_ss, iret->pm_esp);
		if (p >= vmdata->using_jemm_pt_ptr
		    && (p + sizeof(ent->stack)
			< (BYTE *) (EMS_JEMM_FRAME_BASE * PAGE_SIZE))) {
			memcpy(ent->stack, p, sizeof ent->stack);
		} else {
			memset(ent->stack, 0, sizeof ent->stack);
		}
		p = (BYTE *) segoff_to_linear(iret->cs, iret->eip) 
			- (sizeof ent->code) / 2;
		if (p >= vmdata->using_jemm_pt_ptr
		    && (p + sizeof(ent->code)
			< (BYTE *) (EMS_JEMM_FRAME_END * PAGE_SIZE))) {
			memcpy(ent->code, p, sizeof ent->code);
		} else {
			memset(ent->code, 0xEA, sizeof ent->code);
		}
	} else if ((iret->cs & 0x3) != 0) {
		memcpy(&ent->frame, frame, 
		       (unsigned char *) &iret->v86_ds - 
		       (unsigned char *) frame );
	} else {
		WORD ss;
		memcpy(&ent->frame, frame, 
		       (unsigned char *) &iret->pm_esp - 
		       (unsigned char *) frame );
		asm volatile("mov %%ss,%0": "=rm" (ss));
		if (frame->fault & 0x80) {
			ent->frame.code.pm_esp = (DWORD) &iret->pm_esp;
			ent->frame.code.pm_ss = ss;
		} else {
			ent->frame.nocode.pm_esp = (DWORD) &iret->pm_esp;
			ent->frame.nocode.pm_ss = ss;
		}
	}
}

#else /* USE_LOG */

#define lognote(n) /**/
#define lognum(n, m)
#define logpte(n, p) 
#define logclient(n, c, f)
#define logframe(n, f, fl)
#define logmap(p, l)
#define logregs(v, f)

static inline DWORD LTEXT
geteflags_and_cli(void) {
	DWORD eflags;
	asm volatile("pushfl\n\t"
		     "popl %0\n\t"
		     "cli" 
		     : "=r" (eflags));
	return eflags;
}

static inline void LTEXT
seteflags(DWORD eflags) {
	asm volatile("pushl %0\n\t"
		     "popfl"
		     : : "r" (eflags) : "cc");
}

#endif 

/*
 * Get the per VM data for a VM.
 */

static inline struct vm_data * LTEXT
get_vm_data(DWORD vm) {
	if (cb_offset == 0) {
		return NULL;
	}
	return *(struct vm_data **) (vm + cb_offset);
}


/*
 * Allocate per VM data.
 */
static struct vm_data * LTEXT 
alloc_vm_data(DWORD vm) {
	struct vm_data *vmdata;
	if (cb_offset == 0) {
		return NULL;
	}
	vmdata = *(struct vm_data **) (vm + cb_offset);
	if (vmdata == NULL) {
		if (vm_allocate(vm)) {
			return NULL;
		}
		vmdata = *(struct vm_data **) (vm + cb_offset);
	}
	return vmdata;
}

#if 0
int LTEXT
hook_service(DWORD id, HOOKPROC hookproc, int *flag) {
	HOOKPROC dummy;
	if (*flag) {
		return 0;
	}
	if (Hook_Device_Service(id, hookproc, &dummy)) {
		return 1;
	}
	*flag = 1;
	return 0;
}
#endif 

#ifdef USE_LOGRAM

void LTEXT
_use_logram(DWORD UNUSED vm, DWORD UNUSED thread, DWORD UNUSED _vmdata,
	    struct Client_Reg_Struc * crs) {
	DWORD allocation;
	DWORD length;
	DWORD magic;
	DWORD p;
	struct Client_Reg_Struc state;
	DWORD const pagesize = PAGE_SIZE;

	logram_event = NULL;

	Save_Client_State(&state);
	Begin_Nest_V86_Exec();
	crs->Client_AX = 0x7272;
	Exec_Int(0x15);
	allocation = crs->Client_EAX;
	length = crs->Client_EBX;
	magic = crs->Client_ECX;
	lognum("allocation", allocation);
	lognum("length", length);
	lognum("magic", magic);
	End_Nest_Exec();
	Restore_Client_State(&state);

	if (magic != 0x4C4F4752 || length < 1024) {
		return;
	}
	lognum("physpageinf",
	       _GetPhysPageInfo(allocation / pagesize, 
				length / pagesize, 0));
			 
	p = _MapPhysToLinear(allocation, length, 0);
	lognum("linear", p);
	if (p == 0xFFFFFFFF) {
		return;
	}
	*(DWORD *)p = 0;
	log_init((char *)p, length);
	// *(WORD *) 0x472 = 0x4321;
	{
		char buf[32];
		_Sprintf(buf, "logram: %08X\n", p);
		_Trace_Out_Service(buf);
	}
}


Define_RestrictedEventCallback_Thunk(_use_logram);

void LTEXT
use_logram(void) {
	logram_event 
		= Call_Restricted_Event(0, 0, 
					PEF_Wait_For_STI
					| PEF_Wait_Not_Crit
					| PEF_Wait_Not_HW_Int
					| PEF_Wait_Not_Nested_Exec,
					0,
					EventCallback_Thunk(_use_logram),
					0);
	return;
}

#endif /* USE_LOGRAM */


/*
 * Display a blue screen indicating a fatal error and crash the
 * current VM.  If the VMM hasn't be entered, return so that the
 * VMM can be entered and this function called again.
 */

DWORD
fatal_error(struct vm_data *vmdata, char const *logmsg, DWORD logval,
	      char const *msg) {
	unsigned const plen = sizeof "Fatal Error: ";
	static char tmp[256]       = "Fatal Error: ";
	DWORD curvm, vm;

	lognum("fatal", outside_vmm);

	if (logmsg == NULL) {
		lognote("no msg");
	} else {
		lognum(logmsg, logval);
	}

	if (vmdata != NULL) {
		switch_to_v86_pt(vmdata);
		// unforce_jemm_pt(vmdata);
	}

	if (msg != NULL) {
		_Sprintf(tmp + plen - 1,  msg, logval);
	} else if (logmsg != NULL) {
		_Sprintf(tmp + plen - 1, "%s: %04x", logmsg, logval);
	}

	if (outside_vmm > 0) {
		if (outside_vmm > 1 || logmsg == NULL) {
			Crash_Cur_VM();
		}
		return FATAL;
	}

	vm = curvm = Get_Cur_VM_Handle();
	if (vmdata != NULL) {
		vm = vmdata->vm;
	}

	// asm volatile ("sti");

	if (vm != curvm) {
		Fatal_Error_Handler(tmp, 0);
	}

#if 1
	SHELL_SYSMODAL_Message(vm, MB_ASAP | MB_SYSTEMMODAL | MB_NOWINDOW,
			       tmp,  "MyJEMM");
#else
	SHELL_SYSMODAL_Message(vm, MB_SYSTEMMODAL, tmp,  "MyJEMM");
#endif 

	Crash_Cur_VM();
}


/*
 * Handle the case where the VMM wasn't entered when fatal_error()
 * was called.
 */
static int LTEXT
v86_double_fault(struct stack_regs *regs) {
	struct vm_data *vmdata;

	DWORD vm = regs->ebx;
	struct Client_Reg_Struc *crs = (struct Client_Reg_Struc *) regs->ebp;
	if (crs->Client_Error != 1) {
		return 0;
	}

	vmdata = get_vm_data(vm);
	if (vmdata == NULL) {
		return 0;
	}

	fatal_error(NULL, NULL, vm, NULL);

	return 0;
}


HOOKPROC LDATA v86_double_fault_chain;
Define_HookProc_Thunk(v86_double_fault_chain, v86_double_fault);
int LDATA hooked_double_fault = 0;

int LTEXT
hook_double_fault(void) {
	HOOKPROC previous;
	if (hooked_double_fault) {
		return 0;
	}
	hooked_double_fault = 1;
	Hook_V86_Fault(0x08, HookProc_Thunk(v86_double_fault), &previous);
	return 0;
}

int LTEXT
unhook_double_fault(void) {
	if (hooked_double_fault) {
		Unhook_V86_Fault(1, HookProc_Thunk(v86_double_fault));
		hooked_double_fault = 0;
	}
	return 0;
}

/*
 * Cache the linear map to the last CR3 value.
 */
static DWORD last_cr3 = 0;
static DWORD *last_cr3_ptr;

static DWORD *
map_cr3_to_linear(DWORD cr3) {
	DWORD *ptr;

	lognum("cr3", cr3);
	ptr = (DWORD *) _MapPhysToLinear(cr3, PAGE_SIZE, 0);
	lognum("ptr", (DWORD) ptr);
	last_cr3 = cr3;
	last_cr3_ptr = ptr;
	return ptr;
}

/*
 * Get the linear address of the page directory.
 */
static inline DWORD *
get_pd_ptr(void) {
#if 0
	return (DWORD *) 0xFFFBFE000;
#else
	DWORD cr3;
	asm volatile("movl %%cr3,%0": "=r" (cr3));
	cr3 &= PAGE_BASE_MASK;
	if (cr3 == last_cr3) {
		return last_cr3_ptr;
	}
	return map_cr3_to_linear(cr3);
#endif
}

static inline void
aligned_dword_memcpy(DWORD *d, DWORD const *s, unsigned len) {
	asm volatile ("cld; rep movsl"
		      : "+S" (s), "+D" (d), "+c" (len)
		      : : "memory");
}

static inline void
aligned_page_memcpy(DWORD d, DWORD s, unsigned len) {
	DWORD edi = d * PAGE_SIZE;
	DWORD esi = s * PAGE_SIZE;
	DWORD ecx = len * PAGE_SIZE / 4;
	asm volatile ("cld; rep movsl"
		      : "+S" (edi), "+D" (esi), "+c" (ecx)
		      : : "memory");
		      
}

/*
 * Overwrite the VMM's IDT with our own.
 */
int
install_my_idt(DWORD vm) {
	struct idt_entry *idt;
	int i;
	WORD csseg;
	struct cb_s const *cb = (struct cb_s *) vm;
	DWORD eflags;
	int restore_pm_exec = 0;
	struct idtptr idtptr;

	if (installed_my_idt) {
		return 0;
	}
	lognote("install_my_idt");
	if (cb->CB_VM_Status & VMStat_PM_Exec) {
		Set_V86_Exec_Mode();
		restore_pm_exec = 1;
	}
	eflags = geteflags_and_cli();
	asm volatile ("sidt %0": "=m" (idtptr));
	lognum("idt.limit", idtptr.limit);
	lognum("idt.offset", idtptr.offset);
	idt = (struct idt_entry *) idtptr.offset;
	aligned_dword_memcpy((DWORD *) saved_vmm_idt, (DWORD *) idt,
			     IDT_VECTORS * 2);
	asm volatile ("movw %%cs,%0": "=r" (csseg));
	for(i = 0; i < IDT_VECTORS; i++) {
		DWORD v;
		DWORD type = idt[i].flags;
		switch(type) {
		case 0x85: // DPL 0, task gate
			// lognum("task gate", i);
			type = 0x8E;
			changed_idt_type[i] = type;
			break;
		case 0x8F: // DPL 0, trap gate
			// lognum("trap gate", i);
			type = 0x8E;
			changed_idt_type[i] = type;
			break;
		case 0xEF: // DPL 3, trap gate
			// lognum("trap gate EF", i);
			type = 0xEE;
			changed_idt_type[i] = type;
			break;
		}
		switch(type) {
		case 0x8E: // DPL 0, interrupt gate
		case 0xEE: // DPL 3, interrupt gate
#if 1
			v = my_idt_vectors[i];
			idt[i].offset_low = v;
			idt[i].selector = csseg;
			idt[i].flags = type;
			idt[i].offset_hi = v >> 16;
			// v = ((v & 0xFFFF0000) | ((BYTE) type << 8));
			// *(DWORD *) (&idt[i].unused) = v;
#endif
			break;
		default:
			lognum("strange idt", i);
			lognum("type", type);
			break;
		}
	}

	if (old_nmi_handler == NULL) {
		old_nmi_handler = Get_NMI_Handler_Addr();
		Set_NMI_Handler_Addr(nmi_handler);
	}

	installed_my_idt = 1;
	seteflags(eflags);
	if (restore_pm_exec) {
		Set_PM_Exec_Mode();
	}
	return 0;
}

/*
 * Restore the original VMM IDT.
 */
int
restore_vmm_idt(DWORD vm) {
	int restore_pm_exec = 0;
	struct cb_s const *cb = (struct cb_s *) vm;
	DWORD eflags;
	struct idtptr idtptr;
	
	if (!installed_my_idt) {
		return 0;
	}

	eflags = geteflags_and_cli();

	if (cb->CB_VM_Status & VMStat_PM_Exec) {
		lognote("was pm_exec");
		Set_V86_Exec_Mode();
		restore_pm_exec = 1;
	}

	asm volatile ("sidt %0": "=m" (idtptr));
	aligned_dword_memcpy((DWORD *) idtptr.offset, (DWORD *) saved_vmm_idt,
			     IDT_VECTORS * 2);


	if (old_nmi_handler != NULL) {
		Set_NMI_Handler_Addr(old_nmi_handler);
	}

	installed_my_idt = 0;

	if (restore_pm_exec) {
		Set_PM_Exec_Mode();
	}

	seteflags(eflags);

	return 0;
}

/*
 * Fix a protected mode IDT so it's as if it was created based on the
 * original VMM IDT.
 */
int
fix_pm_idt(void) {
	unsigned i;
	struct idt_entry *idt;
	struct idtptr idtptr;
	DWORD eflags;
	eflags = geteflags_and_cli();

	asm volatile ("sidt %0": "=m" (idtptr));
	lognum("idt.limit", idtptr.limit);
	lognum("idt.offset", idtptr.offset);

	idt = (struct idt_entry *) idtptr.offset;

	for(i = 0; i < IDT_VECTORS; i++) {
		DWORD myv = my_idt_vectors[i];
		DWORD winv = idt[i].offset_low | (idt[i].offset_hi << 16);
		if (myv == winv) {
			// lognum("restored", i);
			idt[i] = saved_vmm_idt[i];
		} else if (changed_idt_type[i] != 0
			   && idt[i].flags == changed_idt_type[i]) {
			// lognum("rest flags", i);
			idt[i].flags = saved_vmm_idt[i].flags;
		}
	}
	
	seteflags(eflags);

	return 0;
}

/*
 * Map the VGA page frame at physical address 0xA0000 to the
 * specified address in the JEMM page table.
 */
void
map_vid_mem(struct vm_data *vmdata, DWORD page) {
	unsigned i;
	for(i = 0; i < 0x10; i++) {
		vmdata->jemm_pt[page + i]
			= (0x0A0000 | PAGE_FLAGS) + i * PAGE_SIZE;
	}
}

/*
 * Enable the JEMM memory map.
 */
int
force_jemm_pt(struct vm_data *vmdata, struct init_packet *p) {
	unsigned page;
	DWORD *pd;
	DWORD first, last;
	struct init_packet init = *p; // copy values before mapping DMA pages

	logclient("force_jemm_pt", get_crs(vmdata->vm),
		  vmdata->forced_jemm_pt);

	if (vmdata->forced_jemm_pt) {
		return 0;
	}

	lognum("using_ptr", init.using_jemm_pt_ptr);
	lognum("restore_cbfn", init.restore_jemm_pt_cbfn);
	lognum("dma_buffer", init.dma_buffer);
	lognum("page_frame", init.ems_page_frame);
	lognum("sblive_fix", init.sblive_fix);

	pd = get_pd_ptr();
	if (pd == (DWORD *) 0xFFFFFFFF) {
		return 1;
	}

	lognum("check 0:110", _PageCheckLinRange(0x0, 0x110, 0));
	lognum("check a0:100", _PageCheckLinRange(0xa0, 0x100 - 0xa0, 0));

	first = _GetFirstV86Page();
	last = _GetLastV86Page();

	page = ((DWORD) farptr_to_linear(init.dma_buffer) + PAGE_SIZE - 1)
		/ PAGE_SIZE;
	if (page < first || page > last
	    || (farptr_to_linear(init.dma_buffer) 
		< farptr_to_linear(init.using_jemm_pt_ptr))) {
		lognote("bad dma_buf");
		return 1;
	}

	if (_GetInstanceInfo(page * PAGE_SIZE, PAGE_SIZE) != INSTINFO_NONE) {
		lognote("dma_buf inst");
		return 1;
	}

	lognum("dma page", page);
	vmdata->dma_buffer_page = page;

#if 0
	if (_Assign_Device_V86_Pages(page, 1, vmdata->vm, 0) == 0) {
		lognote("assignv86pgs");
		return 1;
	}
	vmdata->assigned_dma_page = 1;
#endif

#if 1
	vmdata->saved_dma_page = _LinPageLock(page, 2, PAGEMAPGLOBAL);
	if (vmdata->saved_dma_page == 0) {
		lognote("linpglck1");
		unforce_jemm_pt(vmdata);
		return 1;
	}

	if (_MapIntoV86(_GetNulPageHandle(), vmdata->vm, page, 1, 0, 0) == 0) {
		lognote("mapintov86");
		unforce_jemm_pt(vmdata);
		return 1;
	}
	if (_MapIntoV86(_GetNulPageHandle(), vmdata->vm, page + 1, 1, 0, 0)
	    == 0) {
		lognote("mapintov86");
		unforce_jemm_pt(vmdata);
		return 1;
	}
#endif

	if (_LinPageLock(first, last - first + 1, 0) == 0) {
		lognote("linpglck2");
		unforce_jemm_pt(vmdata);
		return 1;
	}
	vmdata->locked_v86 = 1;

	if (vmdata->jemm_pt == NULL) {
		vmdata->jemm_pt = (DWORD *) 
			_PageAllocate(1, PG_SYS, 0,
				      0, 0, 0x100000,
				      &vmdata->jemm_pt_phys,
				      PAGECONTIG | PAGEUSEALIGN | PAGEFIXED);
		lognum("jemm_pt", (DWORD) vmdata->jemm_pt);
		if (vmdata->jemm_pt == NULL) {
			unforce_jemm_pt(vmdata);
			return 1;
		}
	}

	vmdata->using_jemm_pt_ptr
		= (BYTE *) farptr_to_linear(init.using_jemm_pt_ptr);

	vmdata->restore_cbfn = init.restore_jemm_pt_cbfn;
	vmdata->standard_frame_page = init.ems_page_frame * 16U / PAGE_SIZE;
	vmdata->sblive_fix = init.sblive_fix;

	if (_CopyPageTable(0, 0x400, vmdata->jemm_pt, 0) == 0) {
		lognote("_copypagetable");
		unforce_jemm_pt(vmdata);
		return 1;
	}

#if 0
	for(page = EMS_JEMM_FRAME_BASE; page < 0x110; page++) {
		vmdata->jemm_pt[page] = break_page_phys | PAGE_FLAGS;
	}
#endif

	map_vid_mem(vmdata, HMA_PAGE);
	map_vid_mem(vmdata, JEMM_VIDMEM_FIXED_PAGE);

	vmdata->forced_jemm_pt = 1;

#if 0
	for(page = 0; page < 0x400; page++) {
		logpte(page, vmdata->jemm_pt[page]);
	}
#endif

	return 0;
}


/*
 * Disable the JEMM memory map.
 */
int
unforce_jemm_pt(struct vm_data *vmdata) {
	int i;
	int executable = !(((struct cb_s *) vmdata->vm)->CB_VM_Status
			   & VMStat_Not_Executable);

	lognote("unforce_jemm_pt");

	if (vmdata->using_jemm_pt) {
		switch_to_v86_pt(vmdata);
	}

	if (vmdata->locked_v86) {
		DWORD first, last;

		first = _GetFirstV86Page();
		last = _GetLastV86Page();
		_LinPageUnLock(first, last - first + 1, 0);
		vmdata->locked_v86 = 0;
	}

	if (vmdata->saved_dma_page != 0) {
		DWORD dummy;
		if (executable
		    && _LinMapIntoV86(vmdata->saved_dma_page / PAGE_SIZE,
				      vmdata->vm, vmdata->dma_buffer_page,
				      2, 0, &dummy) == 0) {
			return fatal_error(vmdata, "_LinMapIntoV86", 0, NULL);
		}
		_LinPageUnLock(vmdata->saved_dma_page, 2, PAGEMAPGLOBAL);
		vmdata->saved_dma_page = 0;
	}

	if (vmdata->assigned_dma_page) {
		_DeAssign_Device_V86_Pages(vmdata->dma_buffer_page, 1,
					   vmdata->vm, 0);
		vmdata->assigned_dma_page = 0;
	}

	if (!vmdata->forced_jemm_pt) {
		return 0;
	}

	for (i = 0; i < EMS_MAX_HANDLES; i++) {
		if (vmdata->handles[i].pages != NULL) {
			DWORD fatal;
			ems_deallocate_pages(vmdata, &fatal, i + 1);
		}
	}

#if 1
	{
		unsigned page;
		DWORD *p;
		for(page = 0xa0; page < 0x110; page++) {
			logpte(page, vmdata->jemm_pt[page]);
		}
		p = (DWORD *) break_page;
		for(i = 0; i < PAGE_SIZE / 4; i++) {
			if (p[i] != 0xB90FB90F) {
				lognum("break changed", i);
				break;
			}
		}
	}
#endif
	vmdata->forced_jemm_pt = 0;
	return 0;
}


/*
 * Hook the task switch as a sanity check to ensure that the
 * JEMM page tables aren't active when any other VM is running.
 */

void task_switch(DWORD oldvm, DWORD curvm);

Define_TaskSwitchedCallback_Thunk(task_switch);

int LTEXT
hook_task_switch(void) {
	if (hooked_task_switch) {
		return 0;
	}
	hooked_task_switch = 1;
	Call_When_Task_Switched(TaskSwitchedCallback_Thunk(task_switch));
	return 0;
}

 
void LTEXT
task_switch(DWORD oldvm, DWORD curvm) {
	struct vm_data *cvmd = get_vm_data(curvm);
	struct vm_data *ovmd = get_vm_data(oldvm);
	if (cvmd != NULL) {
		// lognum("task in", cvmd->using_jemm_pt);
		if (cvmd->using_jemm_pt) {
			fatal_error(cvmd, "task_in", cvmd->vm, NULL);
		}
	} 
	if (ovmd != NULL) {
		// lognum("task out", ovmd->using_jemm_pt);
		if (ovmd->using_jemm_pt) {
			fatal_error(ovmd, "task_out", ovmd->vm, NULL);
		}
	} 
}


/*
 * Switch to the standard page table.
 */

int
switch_to_v86_pt(struct vm_data *vmdata) {
	DWORD *pd;
	DWORD eflags;

	// lognote("switch_to_v86_pt");

	if (!vmdata->forced_jemm_pt) {
		lognote("not forced");
		return 0;
	}

	if (!vmdata->using_jemm_pt) {
		return 0;
	}

	if (!Test_Cur_VM_Handle(vmdata->vm)) {
		lognote("wrong vm");
		return 1;
	}

	pd = get_pd_ptr();
	if (pd == (DWORD *) 0xFFFFFFFF) {
		return 1;
	}

	eflags = geteflags_and_cli();
	if ((pd[0] & PAGE_BASE_MASK) != vmdata->jemm_pt_phys) {
		seteflags(eflags);
		lognum("pd[0]", pd[0]);
		lognote("no jemm_pt");
		return 1;
	}
	pd[0] = vmdata->save_v86_pde;
	vmdata->using_jemm_pt = 0;
	invalidate_tlb();
	if (vmdata->using_jemm_pt_ptr != NULL) {
		*vmdata->using_jemm_pt_ptr = 0;
	}
	seteflags(eflags);
	return 0;
}

/*
 * Switch to the JEMM page table.
 */

DWORD LTEXT
switch_to_jemm_pt(struct vm_data *vmdata) {
	DWORD *pd;
	DWORD eflags;
	int begin;

	if (!vmdata->forced_jemm_pt
	    || vmdata->using_jemm_pt) {
		return 0;
	}

	// lognote("switch_to_jemm");

	if (!Test_Cur_VM_Handle(vmdata->vm)) {
		lognote("wrong vm");
		return 0;
	}

	pd = get_pd_ptr();
	if (pd == (DWORD *) 0xFFFFFFFF) {
		return fatal_error(vmdata, "no cr3", vmdata->vm, NULL);
	}

	eflags = geteflags_and_cli();

	if ((pd[0] & PAGE_BASE_MASK) == vmdata->jemm_pt_phys) {
		seteflags(eflags);
		lognote("already jemm_pt");
		return 0;
	}

	begin = (DWORD) vmdata->using_jemm_pt_ptr / PAGE_SIZE;

	// lognote("copypt1");
	if (_CopyPageTable(0, 1, vmdata->jemm_pt, 0) == 0) {
		seteflags(eflags);
		return fatal_error(vmdata, "_copypagetable", vmdata->vm, NULL);
	}
	// lognum("copypt2", begin);
	// lognum("end", EMS_JEMM_FRAME_BASE - begin);
#if 0
	if (_CopyPageTable(begin, EMS_JEMM_FRAME_BASE - begin,
			   vmdata->jemm_pt + begin, 0) == 0) {
		seteflags(eflags);
		return fatal_error(vmdata, "_copypagetable", vmdata->vm, NULL);
	}
#endif

	vmdata->save_v86_pde = pd[0];
	pd[0] = vmdata->jemm_pt_phys | PAGE_FLAGS;
	vmdata->using_jemm_pt = 1;
	invalidate_tlb();
	if (vmdata->using_jemm_pt_ptr != NULL) {
		*vmdata->using_jemm_pt_ptr = 1;
	}
	seteflags(eflags);

#if 0
	lognum("old pd[0]", vmdata->save_v86_pde);
	lognum("new pd[0]", pd[0]);
#endif

	return 0;
}

/*
 * Coverted EMS handle structure pointer to and EMS handle (index).
 */
static inline WORD
ems_handle_to_index(struct vm_data *vmdata, struct ems_handle *h) {
	return (h - vmdata->handles) + 1;
}

/*
 * Convert an EMS handle (index) to an EMS handle structure pointer.
 */
static inline struct ems_handle *
ems_index_to_handle(struct vm_data *vmdata, WORD handle) {
	struct ems_handle *h;
	if (handle < 1 || handle > EMS_MAX_HANDLES) {
		return NULL;
	}
	h = vmdata->handles + (handle - 1);
	if (h->pages == NULL) {
		return NULL;
	}
	return h;
}

/*
 * Allocate an EMS handle.
 */
struct ems_handle *
ems_alloc_handle(struct vm_data *vmdata) {
	unsigned i;
	for (i = 0; i < EMS_MAX_HANDLES; i++) {
		if (vmdata->handles[i].pages == NULL) {
			return vmdata->handles + i;
		}
	}
	return NULL;
}

/*
 * Free an EMS handle and the pages it allocated.
 */
void
ems_release_handle(struct vm_data *vmdata, struct ems_handle *h) {
	if (h->pages != NULL) {
		int ret;
		lognum("free pages", (DWORD) h->pages);
		if (vmdata->fh_pages == h->pages) {
			vmdata->fh_pages = 0;
		}
		if (vmdata->mu_pages == h->pages) {
			vmdata->mu_pages = 0;
		}
		ret = _PageFree((HMEM) h->pages, 0);
		lognum("pagefree", ret);
		h->count = 0;
		h->pages = NULL;
	}
}

/*
 * Allocate EMS memory.
 */
WORD
ems_allocate_pages(struct vm_data *vmdata, WORD count, DWORD *handle) {
	struct ems_handle *h;
	if (count == 0) {
		lognote("zero length");
		return EMS_ERROR_ZERO_LENGTH;
	}
	h = ems_alloc_handle(vmdata);
	if (h == NULL) {
		lognote("no handle");
		return EMS_ERROR_NO_HANDLE;
	}
	count *= EMS_PAGE_CONVERT;
	if (count > PAGE_SIZE / 4) {
		// no bigger than one page table worth of pages (4Mb).
		lognote("no free");
		return EMS_ERROR_NO_FREE;
	}
	h->count = count;
#if 0
	h->pages = _PageAllocate(count + 1, PG_SYS, 0,
				 0, 0, 0x1000, NULL,
				 PAGEFIXED | PAGEZEROINIT | PAGEUSEALIGN);
#else
	h->pages = _PageAllocate(count + 1, PG_SYS, 0,
				 0, 0, 0, NULL,
				 PAGEFIXED | PAGEZEROINIT);
#endif
	if (h->pages == NULL) {
		lognote("pageallocate");
		ems_release_handle(vmdata, h);
		return EMS_ERROR_NO_FREE;
	}
	if (_CopyPageTable((DWORD)h->pages / PAGE_SIZE + 1, count,
			   h->pages, 0) == 0) {	
		lognote("copy fail");
		ems_release_handle(vmdata, h);
		return EMS_ERROR_SOFTWARE_ERROR;
	}
	*handle = ems_handle_to_index(vmdata, h);
	return EMS_ERROR_OK;
}

/*
 * Map an EMS page to the standard EMS page frame.
 */
BYTE
ems_standard_map_page(struct vm_data *vmdata, struct ems_handle *h,
		      unsigned phys_page, unsigned logical_page) {
	DWORD dest;
	if (phys_page >= EMS_STANDARD_FRAME_PAGES) {
		lognum("bad stdpg", phys_page);
		return EMS_ERROR_INVALID_PHYS_PAGE;
	}

	if (logical_page != 0xFFFF 
	    && logical_page * EMS_PAGE_CONVERT >= h->count) {
		lognum("bad logical", logical_page);
		return EMS_ERROR_INVALID_PAGE;
	}

	// lognote("standard_map");
	// logmap(phys_page, logical_page);

	dest = vmdata->standard_frame_page + phys_page 
		* EMS_PAGE_CONVERT;

	if (logical_page == 0xFFFF) { /* unmap pages */
		int i;
		HMEM nul = _GetNulPageHandle();
		for(i = 0; i < EMS_PAGE_CONVERT; i++) {
			if (_MapIntoV86(nul, vmdata->vm, dest + i, 1,
					0, 0) == 0) {
				lognote("mapintov86");
				return EMS_ERROR_SOFTWARE_ERROR;
			}
		}
	} else {
		if (_MapIntoV86((HMEM) h->pages, vmdata->vm, dest,
				EMS_PAGE_CONVERT,
				logical_page * EMS_PAGE_CONVERT + 1, 0) == 0) {
			lognote("mapintov86");
			return EMS_ERROR_SOFTWARE_ERROR;
		}
	}
	if (logical_page == 0xFFFF) {
		vmdata->standard_page_map[phys_page].handle = 0;
	} else {
		vmdata->standard_page_map[phys_page].handle
			= ems_handle_to_index(vmdata, h);
	}
	vmdata->standard_page_map[phys_page].page = logical_page;

	return EMS_ERROR_OK;
}

/*
 * Map an EMS page to JEMM page frame.
 */
BYTE
ems_jemm_map_page(struct vm_data *vmdata, struct ems_handle *h,
		  unsigned phys_page, unsigned logical_page,
		  int multiple, int low) {
	DWORD *dest;

	if (vmdata->jemm_pt == NULL) {
		lognote("jemm_pt NULL");
		return EMS_ERROR_SOFTWARE_ERROR;
	}

	if (logical_page != 0xFFFF 
	    && logical_page * EMS_PAGE_CONVERT >= h->count) {
		lognum("bad logical", logical_page);
		return EMS_ERROR_INVALID_PAGE;
	}

	if (phys_page >= EMS_JEMM_FRAME_PAGES / EMS_PAGE_CONVERT) {
		lognum("bad jempg", phys_page);
		return EMS_ERROR_INVALID_PHYS_PAGE;
	}

	if (low) {
		phys_page = phys_page * EMS_PAGE_CONVERT + 1;
	} else {
		struct ems_mapping *map = vmdata->jemm_page_map + phys_page;
		map->handle = ems_handle_to_index(vmdata, h) - 1;
		map->page = logical_page;
		phys_page = phys_page * EMS_PAGE_CONVERT + EMS_JEMM_FRAME_BASE;
	}

	// logmap(phys_page, logical_page);

	dest = vmdata->jemm_pt + phys_page;
	if (logical_page == 0xFFFF) {
		DWORD phys_addr = 0xA0000 | PAGE_FLAGS;
		dest[0] = phys_addr;
		dest[1] = phys_addr;
		dest[2] = phys_addr;
		dest[3] = phys_addr;
	} else {
		DWORD *src;

		logical_page *= EMS_PAGE_CONVERT;
		src = h->pages + logical_page;

		dest[0] = (src[0] & PAGE_BASE_MASK) | PAGE_FLAGS;
		dest[1] = (src[1] & PAGE_BASE_MASK) | PAGE_FLAGS;
		dest[2] = (src[2] & PAGE_BASE_MASK) | PAGE_FLAGS;
		dest[3] = (src[3] & PAGE_BASE_MASK) | PAGE_FLAGS;
	}

	if (!multiple) {
		invalidate_tlb();
	}

	return EMS_ERROR_OK;
}

/*
 * Map multiple EMS pages at once.
 */

DWORD
ems_map_multiple_pages(struct vm_data *vmdata, DWORD *fatal,
		       unsigned subfn, WORD handle,
		       WORD *table, unsigned entries) {
	unsigned i;
	int low;
	struct ems_handle *h = ems_index_to_handle(vmdata, handle);
	int using_jemm_pt;

	if (h == NULL) {
		return EMS_ERROR_INVALID_HANDLE;
	}

	if (subfn == 2) {
		if (!vmdata->using_jemm_pt) {
			return EMS_ERROR_INVALID_SUBFUNCTION;
		}
		low = 1;
	} else if (subfn == EMS_PHYSICAL_PAGE_MODE) {
		low = 0;
	} else {
		return EMS_ERROR_INVALID_SUBFUNCTION;
	}
	
	using_jemm_pt = vmdata->using_jemm_pt;
	if (using_jemm_pt) {
		switch_to_v86_pt(vmdata);
	}

	for(i = 0; i < entries; i++) {
		int ret;
		WORD logc = table[i * 2];
		WORD phys = table[i * 2 + 1];

		if (using_jemm_pt) {
			ret = ems_jemm_map_page(vmdata, h, phys, logc, 1, low);
		} else {
			ret = ems_standard_map_page(vmdata, h, phys, logc);
		}

		if (ret != EMS_ERROR_OK) {
			lognum("map fail", ret);
			return ret;
		}
	}
	if (using_jemm_pt) {
		// switch_to_jemm_pt also flushes TLBs
		if (switch_to_jemm_pt(vmdata) == FATAL) {
			*fatal = FATAL;
		}
	}
	return EMS_ERROR_OK;
}

/*
 * Map an EMS page.
 */
BYTE
ems_map_page(struct vm_data *vmdata,
	     WORD handle, BYTE phys_page, WORD logical_page) {
	struct ems_handle *h = ems_index_to_handle(vmdata, handle);

	if (h == NULL && !(handle == 0 && logical_page == 0xFFFF)) {
		return EMS_ERROR_INVALID_HANDLE;
	}
	if (vmdata->using_jemm_pt) {
		return ems_jemm_map_page(vmdata, h, phys_page,
					 logical_page, 0, 0);
	} else {
		return ems_standard_map_page(vmdata, h, phys_page,
					     logical_page);
	}
}

/*
 * Save the current mapping of the standard EMS page frame.
 */
BYTE
ems_save_page_map(struct vm_data *vmdata, WORD handle) {
	struct ems_handle *h = ems_index_to_handle(vmdata, handle);
	if (h == NULL) {
		return EMS_ERROR_INVALID_HANDLE;
	}
	memcpy(h->saved_map, vmdata->standard_page_map,
	       sizeof h->saved_map);
	return EMS_ERROR_OK;
}

/*
 * Restore the saved mapping of the standard EMS page frame.
 */
BYTE
ems_restore_page_map(struct vm_data *vmdata, DWORD *fatal, WORD handle) {
	unsigned i;
	struct ems_handle *h = ems_index_to_handle(vmdata, handle);
	int using_jemm;

	if (h == NULL) {
		return EMS_ERROR_INVALID_HANDLE;
	}

	using_jemm = vmdata->using_jemm_pt;
	if (using_jemm) {
		switch_to_v86_pt(vmdata);
	}
	for(i = 0; i < 4; i++) {
		struct ems_mapping *p = h->saved_map + i;
		if (ems_map_page(vmdata, p->handle, i, p->page)
		    != EMS_ERROR_OK ) {
			ems_map_page(vmdata,  0, i, 0xFFFF);
		}
	}
	if (using_jemm) {
		if (switch_to_jemm_pt(vmdata) == FATAL) {
			*fatal = FATAL;
		}
	}
	return EMS_ERROR_OK;
}

/*
 * Free previously allocated EMS memory.
 */
BYTE
ems_deallocate_pages(struct vm_data *vmdata, DWORD *fatal, WORD handle) {
	unsigned i;
	struct ems_handle *h = ems_index_to_handle(vmdata, handle);
	int jemm_mode;

	if (h == NULL) {
		return EMS_ERROR_INVALID_HANDLE;
	}

	jemm_mode = vmdata->using_jemm_pt;
	if (jemm_mode) {
		switch_to_v86_pt(vmdata);
	}
	for(i = 0; i < EMS_MAX_HANDLES; i++) {
		if (vmdata->standard_page_map[i].handle == handle) {
			ems_map_page(vmdata, 0, i, 0xFFFF);
		}
	}
	if (jemm_mode) {
		if (switch_to_jemm_pt(vmdata) == FATAL) {
			*fatal = FATAL;
		}
	}
	ems_release_handle(vmdata, h);

	return EMS_ERROR_OK;
}

/*
 * Handle EMS APIs.  Only the EMS APIs actually used are implemented.
 */ 
DWORD LTEXT
ems_api(struct vm_data *vmdata, WORD api, struct frame *frame) {
	BYTE ret = EMS_ERROR_INVALID_FUNCTION;
	DWORD lockable;
	struct iret_frame *iret;
	DWORD fatal = DONT_CHAIN;

	switch(api >> 8) {
	case EMS_GET_PAGE_FRAME:
		lognote("!GET_PAGE_FRAME");
		if (vmdata->using_jemm_pt) {
			frame->ebx = EMS_JEMM_FRAME_BASE * PAGE_SIZE / 16;
		} else {
			frame->ebx = vmdata->standard_frame_page * PAGE_SIZE
				     / 16;
		}
		lognum("frame", frame->ebx);
		ret = EMS_ERROR_OK;
		break;

	case EMS_GET_FREE_PAGE_COUNT:
		lognote("!GET_FREE_COUNT");
		_GetFreePageCount(0, &lockable);
		lockable /= EMS_PAGE_CONVERT;
		if (lockable > EMS_MAX_STDPAGES) {
			lockable = EMS_MAX_STDPAGES;
		}
		frame->ebx = lockable;
		frame->edx = EMS_MAX_STDPAGES;
		lognum("lockable", lockable);
		ret = EMS_ERROR_OK;
		break;

	case EMS_ALLOCATE_PAGES:
		lognote("!ALLOCATE_PAGES");
		lognum("pages",  frame->bx);
		ret = ems_allocate_pages(vmdata, frame->bx, &frame->edx);
		lognum("handle", frame->dx);
		break;
		
	case EMS_DEALLOCATE_PAGES:
		lognote("!DEALLOCATE_PAGES");
		lognum("handle", frame->bx);
		ret = ems_deallocate_pages(vmdata, &fatal, frame->bx);
		if (ret == EMS_ERROR_INVALID_HANDLE) {
			ret = EMS_ERROR_OK; // ignore this error
		}
		break;

	case EMS_MAP_PAGE:
		lognote("!MAP_PAGE");
		// lognum("handle", frame->dx);
		// lognum("phys", frame->al);
		// lognum("logical", frame->bx);
		ret = ems_map_page(vmdata, frame->dx,frame->al, frame->bx);
		break;

	/*
 	 * JEMM extention:
 	 * 	subfunction 0x02: map pages starting at physical addr 0x01000
	 */
	case EMS_MAP_MULTIPLE_PAGES:
		iret = get_iret_frame(frame);

		lognote("!MAP_MULTIPLE_PAGES");
		lognum("subfn", frame->al);
		lognum("handle", frame->dx);
		lognum("ds:si", make_farptr(iret->v86_ds, frame->si));
		lognum("len", frame->cx);

		ret = ems_map_multiple_pages(vmdata, &fatal, 
					     frame->al, frame->dx,
					     segoff_to_linear(iret->v86_ds,
							      frame->si),
					     frame->cx);
		break;

	case EMS_SAVE_PAGE_MAP:
		lognote("!SAVE_PAGE_MAP");
		ret = ems_save_page_map(vmdata, frame->dx);
		break;

	case EMS_RESTORE_PAGE_MAP:
		lognote("!RESTORE_PAGE_MAP");
		ret = ems_restore_page_map(vmdata, &fatal, frame->dx);
		break;

	default:
		lognum("!UNKNOWN EMS", api);
		ret = EMS_ERROR_INVALID_FUNCTION;
		break;
	}
	if (ret != EMS_ERROR_OK) {
		lognum("ems failed", ret);
	}
	frame->eax = ret << 8;
	return fatal;
}

/*
 * Handle VCPI APIs.  Only the "get physical address" API is supported
 * and is used as part of the DMA emulation.  The "set debug registers"
 * API used, but MyJEMM just ignores it.
 */
DWORD
vcpi_api(struct vm_data * UNUSED vmdata, WORD api, struct frame *frame) {
	DWORD ret;
	DWORD page;
	switch(api & 0xFF) {
	case 0x06: /* get physical address */
		page = (WORD) frame->ecx;
		lognum("!GET_PHYS_ADDR", page);
		if (page >= EMS_JEMM_FRAME_BASE && page < EMS_JEMM_FRAME_END) {
			/*   3           2           1         0
			 *  1098 7654 3210 9876 5432 1098 7654 3210
			 *  0000 0000 mmpp pppp ppPP 0000 0000 0000
			 */
			struct ems_mapping *map;

			page -= EMS_JEMM_FRAME_BASE;

			map = vmdata->jemm_page_map + page / EMS_PAGE_CONVERT;
			vmdata->dma_ems_handle = map->handle + 1;
			page = map->page * EMS_PAGE_CONVERT
				+ page % EMS_PAGE_CONVERT;

			frame->edx = (page * PAGE_SIZE) | DMA_MAGIC_ADDRESS;
			lognum("magic", frame->edx);
		} else {
			frame->edx = page * PAGE_SIZE;
		}
		ret = 0;
		break;
	default:
		lognum("!UNKNOWN VCPI", api);
		ret = EMS_ERROR_INVALID_FUNCTION;
		break;
	}

	// lognum("ret", ret);
	frame->eax = ret << 8;

	return 1;
}


/*
 * Remap pages in the HMA in order to implement doubled buffered video
 * page flipping.  An EMS handle hold two back buffers and the VGA
 * memory a physical address 0A0000h is the front buffer.
 */

DWORD LTEXT
jemm_api_MU(struct vm_data *vmdata, struct frame *frame) {
	WORD handle = frame->dx;
	WORD op = frame->cx;
	struct ems_handle *h = ems_index_to_handle(vmdata, handle);
	unsigned page;

	lognum("handle", handle);
	lognum("op", op);

	if (!vmdata->using_jemm_pt) {
		lognote("not jemm_pt");
		frame->eax = 1;
		return DONT_CHAIN;
	}

	switch(op) {
	case 0xFFFF:
		if (h->count != 0x20) {
			frame->eax = 1;
			return DONT_CHAIN;
		}
		vmdata->mu_pages = h->pages;
		break;

	case 0:
		// map phys 0xA0000 to lin 0x100000
		map_vid_mem(vmdata, HMA_PAGE);
		invalidate_tlb();
		break;

	case 1:
	case 2:
		if (h == NULL) {
			lognote("no handle");
			frame->eax = 1;
			return DONT_CHAIN;
		}

		// map EMS pages to lin 0x100000
		if (h->count < 0x10U * op) {
			frame->eax = 1;
			return DONT_CHAIN;
		}
		for(page = 0; page < 0x10; page++) {
			vmdata->jemm_pt[HMA_PAGE + page] 
				= h->pages[(op - 1) * 0x10 + page];
		}
		invalidate_tlb();
		break;

	default:
		frame->eax = 1;
		return DONT_CHAIN;
	}

	frame->eax = 0;
	return DONT_CHAIN;
}

static inline DWORD *
get_vidmem_buf(struct vm_data *vmdata, unsigned buf) {
	if (buf == 0) {
		return (DWORD *) JEMM_VIDMEM_FIXED;
	} 
	return vmdata->mu_pages + (PAGE_SIZE + (buf - 1) * 0x10000) / 4;
}

/*
 * Copy between video buffers.
 *
 * The JEMM version normally does a difference copy, that is, 
 * only copies the bytes that different between the old back buffer
 * and the new back buffer to the front buffer (video RAM).
 * This routine assumes that this optimization isn't worth doing
 * on modern computers and just copies the entire buffer.
 */

DWORD LTEXT
jemm_api_CU(struct vm_data *vmdata, struct frame *frame) {
	unsigned src = frame->cl;
	unsigned dest = frame->ch;
	DWORD *s, *d;
	// unsigned diffcopy = frame->dx;

	lognum("src", src);
	lognum("dest", dest);
	lognum("diffcopy", frame->dx);

	if (!vmdata->using_jemm_pt 
	    || dest > 2 || src > 2
	    || vmdata->mu_pages == NULL) {
		frame->eax = 1;
		return DONT_CHAIN;
	}
	
	s = get_vidmem_buf(vmdata, src);
	d = get_vidmem_buf(vmdata, dest);

	aligned_dword_memcpy(d, s, 320 * 200 / 4);

	frame->eax = 0;

	return DONT_CHAIN;
}

/*
 * Map an EMS handle's memory to a fixed linear address about 1Mb.
 * Currently unused.
 */
DWORD LTEXT
jemm_api_FH(struct vm_data *vmdata, struct frame *frame) {
	WORD handle = frame->dx;
	struct ems_handle *h = ems_index_to_handle(vmdata, handle);

	if (h == NULL) {
		frame->eax = 1;
		return DONT_CHAIN;
	}

	vmdata->fh_pages = h->pages;
	frame->edx = JEMM_FH_BASE;
	frame->eax = 0;

	return DONT_CHAIN;
}	

static inline void *
check_linmap(struct vm_data *vmdata, DWORD lin, DWORD length) {
	if (lin + length < 0x110000) {
		return (void *) lin;
	}
	if (lin >= JEMM_FH_BASE 
	    && lin + length <= (JEMM_FH_BASE + PAGE_SIZE * 2)
	    && vmdata->fh_pages != NULL) {
		return (void *) (vmdata->fh_pages + PAGE_SIZE 
				 + (lin - JEMM_FH_BASE));
	}
	return 0;
}

/*
 * Implement a protected mode callback with segment descriptor limits
 * set 4Gb instead of 64k by implementing function being called,
 * rather than doing the actual callback.
 * Currently unused, and will cause a fatal error if called.
 */

DWORD LTEXT
jemm_api_SP(struct vm_data *vmdata, struct frame *frame) {
	struct iret_frame *iret = get_iret_frame(frame);
	// WORD cb_seg = iret->v86_es;
	// WORD cb_off = frame->dx;
	DWORD *bp = segoff_to_linear(iret->pm_ss, frame->bp);

	DWORD length = bp[0x0E];
	void *dest = check_linmap(vmdata, bp[0x06], length);
	void *src = check_linmap(vmdata, bp[0x0A], length);

	lognum("dest", (DWORD) dest);
	lognum("src", (DWORD) src);
	lognum("length", length);

	fatal_error(vmdata, "SP call", frame->dx, "SP called");

	memcpy(dest, src, length);

	return DONT_CHAIN;
}

/*
 * Handle JEMM API functions.
 */
DWORD LTEXT
jemm_api(struct vm_data *vmdata, WORD api, struct frame *frame) {
	WORD seg, off;
	struct iret_frame *iret = get_iret_frame(frame);

	switch(api) {
	case 0x4143: // 'AC' install check
		lognote("@INST CHECK");
		frame->eax = 0;
		frame->ebx = 0x1209;
		frame->ecx = 6;		// magic checksum
		return DONT_CHAIN; 

	case 0x1234:
		lognote("old switch");

	case 0x534d: // 'SM' switch to jemm page tables
		lognote("@SWITCH_TO_JEMM_PT");

		frame->bx = vmdata->using_jemm_pt;
		if (switch_to_jemm_pt(vmdata) == FATAL) {
			return FATAL;
		}
		return DONT_CHAIN;

	case 0x736d: // 'sm' switch to standard page tables
		lognote("@SWITCH_TO_V86_PT");
		frame->ebx = vmdata->using_jemm_pt;
		switch_to_v86_pt(vmdata);
		return DONT_CHAIN;; 

	case 0x5346: // 'SF' set client fault handler
		lognum("@SET_CLIENT_FAULT", make_farptr(iret->v86_es,
							frame->di));
		frame->eax = 0;
		return DONT_CHAIN;

	case 0x4646: // 'FF' get using_jemm_pt pointer
		lognote("GET USING_PTR");
		seg = (DWORD) vmdata->using_jemm_pt_ptr / 16;
		off = (DWORD) vmdata->using_jemm_pt_ptr % 16;
		iret->v86_es = seg;
		frame->eax = 0;
		frame->edi = off;
		lognum("ptr", make_farptr(seg, off));
		return DONT_CHAIN;

	case 0x4753: // 'GS' get page table status;
		lognote("@GET PT STATUS");
		frame->eax = 1;
		return DONT_CHAIN;

	case 0x5352: // 'SR' set page memmap table
		lognote("@SET PAGE MEMMAP");
		return DONT_CHAIN;

	case 0x4d55: // 'MU' 
		lognote("@SET VIDBUF");
		return jemm_api_MU(vmdata, frame);

	case 0x4355: // 'CU'
		lognote("@COPY VIDBUF");
		return jemm_api_CU(vmdata, frame);

	case 0x554d: // 'UM'
		lognote("@GET UMB SEG");
		frame->eax = 0;
		return DONT_CHAIN;

	case 0x4648: // 'FH'
		lognote("@MAP EXT LIN");
		return jemm_api_FH(vmdata, frame);

	case 0x5350: // 'SP'
		lognote("@FLATREAL CALL");
		return jemm_api_SP(vmdata, frame);

	case 0x584d: // 'XM'
		lognote("@MEM SIZE");
		frame->eax = MEMORY_SIZE / PAGE_SIZE;
		return DONT_CHAIN;

	case 0x5650: // 'VP'
		lognote("@VERIFY CHECKSUM");
		iret->eip += 2;
		return DONT_CHAIN;

	case 0x5645: // 'VE'
		lognote("@VERSION");
		frame->ax = 0x499;
		return DONT_CHAIN;

		// 'CH'
		// 'AF'
		// 'EI'
	       
	case 0x554e: // 'UN' enter real mode
	default:
		return fatal_error(vmdata, "unknown jemm", api,
				   "Unknown JEMM API function called: %04x");
		// lognum("@UNKNOWN JEMM", api);
		// return DONT_CHAIN;
	}
	return DONT_CHAIN;
}

/*
 * Handle faults from V86 mode.
 */
DWORD LTEXT
handle_fault(struct vm_data *vmdata, struct frame *frame) {
	DWORD fault;

	fault = frame->fault & 0x7F;
	// logframe("handle_fault", frame, vmdata->using_jemm_pt);

	if (fault == 0xe) { // page fault
		DWORD cr2;
		asm volatile("movl %%cr2,%0": "=r" (cr2));
		if (cr2 >= 2*PAGE_SIZE && cr2 < 0xc0000) {
			logframe("handle_fault", frame, vmdata->using_jemm_pt);
			lognum("cr2", cr2);
			lognum("code", frame->code.error_code);
		}
		if (vmdata->using_jemm_pt 
		    && cr2 >= EMS_JEMM_FRAME_BASE * PAGE_SIZE
		    && cr2 < EMS_JEMM_FRAME_END * PAGE_SIZE) {
			logregs(vmdata, frame);
			return fatal_error(vmdata, "jemm pgfault", vmdata->vm,
					   NULL);
		}
	} else if (fault == 0xd) { // general protection fault
		struct iret_frame *iret = get_iret_frame(frame);
		logframe("handle_fault", frame, vmdata->using_jemm_pt);
		lognum("eflags", iret->eflags);
		lognum("insn", *(DWORD *) segoff_to_linear(iret->cs,
							   iret->eip));
	}

	if (vmdata->using_jemm_pt) {
		switch_to_v86_pt(vmdata);
		return vmdata->restore_cbfn;
	}
	return 0;
}

static inline BYTE
byteoff(DWORD num, unsigned off) {
	return (BYTE) (num >> (off * 8));
}

static inline int
is_32bit_insn(DWORD insn) {
	if ((insn & 0xFF) == 0x66
	    || (insn & 0xFF00) == 0x6600
	    || (insn & 0xFF0000) == 0x660000
	    || (insn & 0xFF000000) == 0x66000000) {
		return 1;
	}
	return 0;
}

/*
 * Handle sound card DMA from EMS mapped memory.
 */
int
figure_linear(struct vm_data *vmdata, struct dma_data *dma,
	      BYTE *plinear) {
	DWORD linear;
	DWORD page;

	linear = (dma->addr[2] << 16) 
		| (dma->addr[1] << 8) 
		| dma->addr[0];

	lognum("dma linear", linear);
	
	if ((linear & DMA_MAGIC_MASK) == DMA_MAGIC_ADDRESS) {
		int using_jemm_pt;
		struct ems_handle *h 
			= ems_index_to_handle(vmdata, vmdata->dma_ems_handle);

		if (h == NULL) {
			return fatal_error(vmdata, "no dma handle", 0, NULL);
		}

#if 0
		if (dma->len[1] * 256 + dma->len[0]
		    + linear % PAGE_SIZE > PAGE_SIZE) {
			lognum("bad dma range",
			       dma->len[1] * 256 + dma->len[0]);
			return 1;
		}
#endif

		page = ((linear & ~DMA_MAGIC_MASK) / PAGE_SIZE);
		// h = vmdata->handles + 0; /* must be first handle */

		if (page >= h->count) {
			lognum("bad dma page", page);
			return 1;
		}

		page = (DWORD) h->pages / PAGE_SIZE + 1 + page;

		lognum("dma sys page", page);

		using_jemm_pt = vmdata->using_jemm_pt;
		if (using_jemm_pt) {
			switch_to_v86_pt(vmdata);
		}

#if 1
		if (_LinMapIntoV86(page, vmdata->vm,
				   vmdata->dma_buffer_page
				   + vmdata->dma_buffer_index, 
				   1, 0, &page) == 0) {
			lognum("linmapintov86", 
			       vmdata->dma_buffer_page);
			return 1;
		}
#else
		aligned_page_memcpy(vmdata->dma_buffer_page
				    + vmdata->dma_buffer_index, page, 1);
		page = vmdata->dma_buffer_page + vmdata->dma_buffer_indez;
#endif

		vmdata->dma_buffer_index = !vmdata->dma_buffer_index;

		if (using_jemm_pt) {
			if (switch_to_jemm_pt(vmdata) == FATAL) {
				return FATAL;
			}
		}

		lognum("mapped page", page);
		linear = (page * PAGE_SIZE) | (linear % PAGE_SIZE);
		
		lognum("new linear", linear);
		// lognum("length", dma->len[1] * 256 + dma->len[0]);
	}

	if (linear == 0) {
		linear = vmdata->dma_buffer_page * PAGE_SIZE;
	}

	plinear[0] = byteoff(linear, 0);
	plinear[1] = byteoff(linear, 1);
	plinear[2] = byteoff(linear, 2);

	return 0;
}

static BYTE const dma_pagereg_map[] = { 0x87, 0x83, 0x81, 0x82 };

/*
 * Emulate setting sound card DMA address.
 */

DWORD LTEXT
handle_dma_addr(struct vm_data *vmdata, int channel, BYTE al) {
	struct dma_data *dma;
	BYTE linear[4];
	DWORD r;

	dma = vmdata->dma + channel;
	dma->addr[vmdata->dma_flipflop] = al;
	vmdata->dma_flipflop = !vmdata->dma_flipflop;
	
	if (vmdata->dma_flipflop != 0) {
		return DONT_CHAIN; // handled
	}

	r = figure_linear(vmdata, dma, linear);
	if (r != 0) {
		return r;
	}

	_Simulate_VM_IO(dma_pagereg_map[channel], Byte_Output, linear[2]);
	dma->addr[2] = linear[2];
	_Simulate_VM_IO(0x0C, Byte_Output, 00);
	_Simulate_VM_IO(channel * 2, Byte_Output, linear[0]);
	dma->addr[0] = linear[0];
	if (dma->addr[1] == linear[1]) {
		return CHAIN; // pass to vmm
	}
	_Simulate_VM_IO(channel * 2, Byte_Output, linear[1]);
	dma->addr[1] = linear[1];
	return DONT_CHAIN; // handled
}

inline static int LTEXT
handle_dma_page(struct vm_data *vmdata, int channel, BYTE al) {
	struct dma_data *dma = vmdata->dma + channel;

	dma->addr[2] = al;
	return CHAIN;
}

/*
 * Emulate DMA registers for sound card DMA.
 */
DWORD LTEXT
handle_dma(struct vm_data *vmdata, WORD port, int input, int size,
	   struct frame *frame) {
	BYTE al;

	if (size != 1) {
		lognote("bad dma type");
		return 1;
	}

	if (input) {
		lognote("DMA IO in");
		return CHAIN;
	}

	al = (BYTE) frame->eax;
	lognum("DMA IO out", al);

	switch(port) {
	case 0x00:
		return handle_dma_addr(vmdata, 0, al);

	case 0x01:
		vmdata->dma[0].len[vmdata->dma_flipflop] = al;
		vmdata->dma_flipflop = !vmdata->dma_flipflop;
		return CHAIN; // pass to vmm

	case 0x02:
		return handle_dma_addr(vmdata, 1, al);

	case 0x03:
		vmdata->dma[1].len[vmdata->dma_flipflop] = al;
		vmdata->dma_flipflop = !vmdata->dma_flipflop;
		return CHAIN;

	case 0x04:
		return handle_dma_addr(vmdata, 2, al);

	case 0x05:
		vmdata->dma[2].len[vmdata->dma_flipflop] = al;
		vmdata->dma_flipflop = !vmdata->dma_flipflop;
		return CHAIN;

	case 0x06:
		return handle_dma_addr(vmdata, 3, al);

	case 0x07:
		vmdata->dma[3].len[vmdata->dma_flipflop] = al;
		vmdata->dma_flipflop = !vmdata->dma_flipflop;
		return CHAIN;

	case 0x0A:
		if (al & 0x04) {
			return 0; // mask dma
		}
		if (vmdata->sblive_fix) {
			int channel = al & 0x3;
			struct dma_data *dma = vmdata->dma + channel;
			lognum("dma unmask", channel);
			lognum("address", (dma->addr[0] 
					   + dma->addr[1] * 256 
					   + dma->addr[2] * 65536));
			lognum("length", dma->len[0] + dma->len[1] * 256);
			_Simulate_VM_IO(0x20, Byte_Output, 0x20);
		}
		return CHAIN;

#if 0
	case 0x0B:
		if ((al & 0x0C) != 0x08) {
			lognum("bad dma mode", al);
			return 1;
		}
		vmdata->dma[al & 0x03].mode = al;
		break;
#endif

	case 0x0C:
		vmdata->dma_flipflop = 0;
		return CHAIN; // chain to vmm

	case 0x87: // channel 0 hi
		return handle_dma_page(vmdata, 0, al);

	case 0x83: // channel 1 hi
		return handle_dma_page(vmdata, 1, al);

	case 0x81: // channel 2 hi
		return handle_dma_page(vmdata, 2, al);

	case 0x82: // channel 3 hi
		return handle_dma_page(vmdata, 3, al);

	default:
		return CHAIN;
	}
	return CHAIN; // shouldn't get here
}

static inline void
log_client_eflags(struct vm_data *vmdata) {
	lognum("CRS.eflags", get_crs(vmdata->vm)->Client_EFlags);
}


/*
 * Handle I/O operations from the V86 task.
 */

DWORD LTEXT
handle_io(struct vm_data *vmdata,
	  DWORD insn, DWORD insn2, DWORD eip, WORD opcode,
	  struct frame *frame) {
	struct iret_frame *iret = get_iret_frame(frame);
	DWORD cbfn;
	WORD *p;
	int port;
	WORD *v86_stack;
	DWORD temp_eflags;
	int input = 1;
	int using_jemm_pt = vmdata->using_jemm_pt;
	char const *msg = NULL;
	int quiet = 0;

	port = -1;
	cbfn = vmdata->restore_cbfn - 6;

	switch(opcode & 0xFF) {
	case 0x6C:
	case 0x6D:
		port = (WORD) frame->edx;
		msg = "INS PORT";
		input = 3;
		break;

	case 0x6E:
	case 0x6F:
		port = (WORD) frame->edx;
		msg = "OUTS PORT";
		input = 2;
		break;

	case 0xE4:
	case 0xE5:
		port = (opcode >> 8) & 0xFF;
		msg = "IN PORT";
		input = 1;
		break;

	case 0xE6:
	case 0xE7:
		port = (opcode >> 8) & 0xFF;
		msg = "OUT PORT";
		input = 0;
		break;

	case 0xEC:
	case 0xED:
		port = (WORD) frame->edx;
		msg = "IN PORT";
		input = 1;
		break;

	case 0xEE:
	case 0xEF:
		port = (WORD) frame->edx;
		msg = "OUT PORT";
		input = 0;
		break;
	}

	if (port == 0x201 && input) {
		// joystick polling
		quiet = 1;
	} else if (port == 0x20 && !input && (frame->eax & 0xFF) == 0x20) {
		// lognote("PIC1 EOI");
#if 0
		DWORD status;
		if (VPICD_Get_IRQ_Complete_Status(7, &status)) {
			lognum("IRQ7", status);
			if (status == 0x23) {
				logframe("handle_io", frame, vmdata->using_jemm_pt);
				log_client_eflags(vmdata);
#if 0
				_Simulate_VM_IO(0x20, Byte_Output, 0x20);
				VPICD_Get_IRQ_Complete_Status(7, &status);
				lognum("after EOI", status);
				logframe("handle_io", frame, vmdata->using_jemm_pt);
				log_client_eflags(vmdata);
				return DONT_CHAIN;
#endif
			}
		}

#endif
		quiet = 1;
	}
		
	if (!quiet) {
#if 0
		logframe("handle_io", frame, vmdata->using_jemm_pt);
		lognum("opcode", (WORD) opcode);
		lognum("insn", insn);
		lognum("insn2", (WORD) insn2);
		if (msg != NULL) {
			lognum(msg, port);
		}
#endif
	}
		
	switch(opcode & 0xFF) {
	case 0x9C:
		// logregs(vmdata, frame);
		if (is_32bit_insn(insn)) {
			lognote("PUSHFD");
			insn2 = (insn2 & 0x00FF) | 0xCC00;
		} else {
			lognote("PUSHF");
			insn2 = 0x90CC;
		}
		break;

	case 0x9D:
		if (!using_jemm_pt) {
			lognote("POPF");
			return CHAIN;
		}
		// logregs(vmdata, frame);
		iret->pm_esp -= 4;
		v86_stack = (WORD *) segoff_to_linear(iret->pm_ss,
						      iret->pm_esp);
		if (is_32bit_insn(insn)) {
			lognote("POPFD");
			v86_stack[0] = v86_stack[2];
			v86_stack[1] = v86_stack[3];
			v86_stack[2] = eip;
			v86_stack[3] = iret->cs;
		} else {
			lognote("POPF");
			v86_stack[0] = v86_stack[2];
			v86_stack[1] = eip;
			v86_stack[2] = iret->cs;
		}
		eip = cbfn & 0xFFFF;
		iret->cs = cbfn >> 16;
		cbfn = 0; /* chain, restore_cbfn already on stack */
		break;

	case 0xFA:
		lognote("CLI");
		break;

	case 0xFB:
		lognote("STI");
		break;

	case 0xCF:
		lognote("IRET");
		if (!using_jemm_pt) {
			return CHAIN;
		}
		if (is_32bit_insn(insn)) {
			return fatal_error(vmdata, "iret32", vmdata->vm, NULL);
		}
		iret->pm_esp -= 4;
		v86_stack = (WORD *) segoff_to_linear(iret->pm_ss,
						      iret->pm_esp);
		temp_eflags = v86_stack[4];
		v86_stack[4] = v86_stack[3];
		v86_stack[3] = v86_stack[2];
		v86_stack[2] = temp_eflags;
		v86_stack[1] = vmdata->restore_cbfn >> 16;
		v86_stack[0] = vmdata->restore_cbfn & 0xFFFF;
		eip = cbfn & 0xFFFF;
		iret->cs = cbfn >> 16;
		cbfn = 0; /* chain, restore_cbfn already on stack */
		break;

	default:
		if (port != -1) {
			break;
		}
		lognum("BAD I/O", opcode);
		lognum("insn", insn);
		lognum("insn2", insn2);
#if 1
		return fatal_error(vmdata, "bad i/o", opcode, NULL);
#endif
		break;
	}

	if (port != -1) {
		if (port <= 0x0F
		    || (port >= 0x80 && port <= 0x8F)
		    || (port >= 0xC0 && port <= 0xDF)) {
			int size = 1;
			DWORD ret;
			if (opcode & 1) {
				if (is_32bit_insn(insn)) {
					size = 4;
				}
				size = 2;
			}
			log_client_eflags(vmdata);
#if 0
			lognote("dma attempted");
			logregs(vmdata, frame);
#endif
			ret = handle_dma(vmdata, port, input, size, frame);
			if (ret == DONT_CHAIN) {
				iret->eip = eip;
				return ret;
			}
			if (ret == FATAL) {
				return ret;
			}
			if (ret != CHAIN) {
				// switch_to_v86_pt(vmdata);
				// _Debug_Out_Service("bad dma\n");
				return fatal_error(vmdata, "bad dma",
						   vmdata->vm, NULL);
			}
		}
	}

	if (!input && !quiet) {
#if 0
		lognum("eax", frame->eax);
#endif
	}

	if (!using_jemm_pt) {
		return CHAIN;
	}

	switch_to_v86_pt(vmdata);

	p = farptr_to_linear(vmdata->restore_cbfn - 6);
	*(DWORD *)p = insn;
	p[2] = (WORD) insn2;

	iret->eip = eip;

	return cbfn;
}

/*
 * Handle certain software interrupts that need special handling.
 */
DWORD LTEXT
handle_funky_swint(struct vm_data *vmdata,
		   DWORD intr, struct frame *frame) {
	struct iret_frame *iret = get_iret_frame(frame);
	WORD *v86_stack;
	DWORD vec;

	logframe("handle_funky", frame, vmdata->using_jemm_pt);

	iret->pm_esp -= 6;
 	v86_stack = (WORD *) segoff_to_linear(iret->pm_ss, iret->pm_esp);

	v86_stack[0] = iret->eip;
	v86_stack[1] = iret->cs;
	v86_stack[2] = iret->eflags;

#if 1
	switch_to_v86_pt(vmdata);
	vec = Get_Instanced_V86_Int_Vector(intr, vmdata->vm);
	if (switch_to_jemm_pt(vmdata) == FATAL) {
		return FATAL;
	}
#else
	vec = ((DWORD *) 0)[intr];
#endif

	iret->eip = vec & 0xFFFF;
	iret->cs = vec >> 16;
	iret->eflags &= 0xFFFFFCFF;

	return DONT_CHAIN;
}

/*
 * Handle breakpoints.  Breakpoints in the callback routine indicate
 * mean switch back to JEMM page tables, and possible special handling.
 */
static inline DWORD
handle_breakpoint(struct vm_data *vmdata, struct frame *frame UNUSED,
		  struct iret_frame *iret) {
	DWORD loc = make_farptr(iret->cs, iret->eip);
	WORD *v86_stack = (WORD *) segoff_to_linear(iret->pm_ss,
						    iret->pm_esp);
	switch(loc - vmdata->restore_cbfn) {
	case +1:
		// lognote("restore_cbfn");
		iret->eip = v86_stack[0];
		iret->cs = v86_stack[1];
		iret->pm_esp += 4;
		break;

	case 0:
		lognote("cbfn PUSHFD");
		iret->cs = v86_stack[3];
		iret->eip = v86_stack[2];
		v86_stack[3] = v86_stack[1];
		v86_stack[2] = v86_stack[0];
		iret->pm_esp += 4;
		// logregs(vmdata, frame);
		break;

	case -1:
		lognote("cbfn PUSHF");
		iret->cs = v86_stack[2];
		iret->eip = v86_stack[1];
		v86_stack[2] = v86_stack[0];
		iret->pm_esp += 4;
		// logregs(vmdata, frame);
		break;

	default:
		return fatal_error(vmdata, "breakpt", loc, NULL);
	}

	if (switch_to_jemm_pt(vmdata) == FATAL) {
		return FATAL;
	}

	return DONT_CHAIN;
}

/*
 * Handle software interrupt in the V86 task.
 */

DWORD LTEXT
handle_swint(struct vm_data *vmdata, DWORD intr, struct frame *frame) {
	struct iret_frame *iret;
	DWORD loc;
	
	iret = get_iret_frame(frame);
	loc = make_farptr(iret->cs, iret->eip);
       
	if ((frame->fault & 0x7F) == 0xd 
	    && make_farptr(frame->code.cs, frame->code.eip)
	       == vmdata->restore_cbfn) {
		logframe("handle_swint", frame, vmdata->using_jemm_pt);
		return fatal_error(vmdata, "recursion", loc, NULL);
	}

	if (frame->fault == 3) {
		// logframe("handle_break", frame, vmdata->using_jemm_pt);
		return handle_breakpoint(vmdata, frame, iret);
	} else if (intr == 1) {
		logframe("handle_swint", frame, vmdata->using_jemm_pt);
		return fatal_error(vmdata, "trace", loc, NULL);
	} else if (intr == 0x67) {
		if ((WORD) frame->eax >= 0x4000
		    && (WORD) frame->eax <= 0x5DFF) {
			// logframe("handle_swint", frame, vmdata->using_jemm_pt);
			return ems_api(vmdata, (WORD) frame->eax, frame);
		} else if ((frame->eax & 0xFF00) == 0xDE00
			   /* && vmdata->using_jemm_pt */) {
			// logframe("handle_swint", frame, vmdata->using_jemm_pt);
			vcpi_api(vmdata, (WORD) frame->eax, frame);
			return DONT_CHAIN;
		}
		lognum("int67", frame->eax);
	} else if (intr == 0x15
		   && (WORD) frame->eax == 0x1209) {
		// logframe("handle_swint", frame, vmdata->using_jemm_pt);
		return jemm_api(vmdata, (WORD) frame->ebx, frame);
	} else if (intr == 0x15) {
		lognum("int15", frame->eax);
	}

	if (intr < 16) {
		// logframe("handle_swint", frame, vmdata->using_jemm_pt);
		// logregs(vmdata, frame);


		if (iret->cs == (vmdata->restore_cbfn >> 16)
		    && (WORD) iret->eip 
		       > (DWORD) (WORD) vmdata->restore_cbfn + 32) {
			return fatal_error(vmdata, "overrun", (WORD) iret->eip,
					   NULL);
		}
#if 0
		if (intr != frame->fault) {
			lognum("intr", intr);
		}
		lognum("insn", *(DWORD *) segoff_to_linear(iret->cs,
							   iret->eip));
#endif
	}
#if 0
	else if (1 || vmdata->using_jemm_pt) {
		logframe("handle_swint", frame, vmdata->using_jemm_pt);
		logregs(vmdata, frame);
	}
#endif

	if (intr == 0x21 && frame->ah == 0x4B) {
		BYTE *p = segoff_to_linear(iret->v86_es, frame->bx);
		char *arg = farptr_to_linear(*(DWORD *) (p + 2));
		lognote("dos exec");
		lognote(segoff_to_linear(iret->v86_ds, frame->dx));
		lognum("arg len", *arg);
		lognote(arg + 1);
	}

	if (vmdata->using_jemm_pt) {
		DWORD cbfn;
		if (intr >= 0x34 && intr <= 0x3f) {
			return handle_funky_swint(vmdata, intr, frame);
		}
		switch_to_v86_pt(vmdata);

		cbfn = vmdata->restore_cbfn;
		if ((frame->fault & 0x7F) == 0x0D) {
			WORD *p = farptr_to_linear(cbfn - 2);
			*p = 0xCD | (intr << 8);
			cbfn -= 2;
		}
		return cbfn;
	}
	return CHAIN;
}

struct call_packet {
	DWORD function;
	DWORD *args;
	struct vm_data *vmdata;
};

/*
 * Call the appropriate handler for the type of interrupt 
 * received by the idt.asm.
 */
void
call_handler(DWORD _packet) {
	struct call_packet *packet = (struct call_packet *) _packet;

	switch(packet->function) {
	case 0: // swint
		packet->function 
			= handle_swint(packet->vmdata,
				       packet->args[0], 
				       (struct frame *) packet->args[1]);
		break;

	case 1: // fault
		packet->function 
			= handle_fault(packet->vmdata,
				       (struct frame *) packet->args[0]);
		break;

	case 2: // io	
		packet->function
			= handle_io(packet->vmdata,
				    packet->args[0],
				    packet->args[1],
				    packet->args[2],
				    packet->args[3],
				    (struct frame *) packet->args[4]);
		break;

	default:
		packet->function 
			= fatal_error(packet->vmdata,
				      "call_handler", packet->vmdata->vm,
				      NULL);
		break;
	}
}

/*
 * Handle an interrupt received in idt.asm.
 */

DWORD
handle_idt(DWORD function, DWORD args) {
	struct vm_data *vmdata;
	struct call_packet packet;
	DWORD count = 0;

	vmdata = get_vm_data(Get_Cur_VM_Handle());

	asm("xchg %0,%1": "+m" (nmi_counter), "+r" (count));

	if (count != 0) {
		lognum("nmi_count", count);
	}

	count = 0;
	asm("xchg %0,%1": "+m" (nmi_counter2), "+r" (count));
	if (count != 0) {
		lognum("nmi_count2", count);
	}

	if (vmdata == NULL || !vmdata->forced_jemm_pt) {
		return CHAIN; // chain
	}

	if (function == 3) {
		DWORD *pd;

		if (!vmdata->using_jemm_pt) {
			return 0;
		}

		pd = last_cr3_ptr;
		// eflags = geteflags_and_cli();

		if ((pd[0] & PAGE_BASE_MASK) == vmdata->jemm_pt_phys) {
			pd[0] = vmdata->save_v86_pde;
			vmdata->using_jemm_pt = 0;
			if (vmdata->using_jemm_pt_ptr != NULL) {
				*vmdata->using_jemm_pt_ptr = 0;
			}
			invalidate_tlb();
		}
		// seteflags(eflags);
		return vmdata->restore_cbfn;
	}

	if (vmdata->using_my_stack) {
		return fatal_error(vmdata, "using_my_stack", vmdata->vm, NULL);
	}

	packet.function = function;
	packet.args = &args;
	packet.vmdata = vmdata;
	vmdata->using_my_stack = 1;
	_Call_On_My_Stack(call_handler, (DWORD) &packet,
			  (BYTE *) vmdata->my_stack + sizeof vmdata->my_stack,
			  sizeof vmdata->my_stack);
	vmdata->using_my_stack = 0;

	return packet.function;
}

#if 0

void 
crashme(DWORD *stack UNUSED) {
	return fatal_error(get_vm_data(Get_Cur_VM_Handle()),
			   "crashme", 0, NULL);
}

#endif

/*
 * Free up any resources used by a vm_data node.
 */

int LTEXT
vmdata_destroy_node(struct vm_data *node, int UNUSED pagesafe) {
	struct vm_data *p;
	int ret = 0;
	DWORD cur_vm = Get_Cur_VM_Handle();
	DWORD sys_vm = Get_Sys_VM_Handle();

	lognum("destroy_node", (DWORD) vmdata_list);

	if (vmdata_list == NULL) {
		return node != NULL;
	}

	if (List_Get_First(vmdata_list, (void **) &p)) {
		return 0;
	}

	do {
		if (node == NULL || p == node) {
			p->using_jemm_pt_ptr = NULL;
			p->restore_cbfn = NULL;
			unforce_jemm_pt(p);
			if (p->jemm_pt != NULL) {
				_PageFree((HMEM) p->jemm_pt, 0);
				p->jemm_pt = NULL;
			}
			if (p->vm != 0) {
				*(struct vm_data **) (p->vm 
						      + cb_offset) = NULL;
			}
			if (node == NULL && p->vm != 0
			    && p->vm != cur_vm && p->vm != sys_vm) {
				Nuke_VM(p->vm);
			}
			p->vm = 0;
			if (node != NULL) {
				if (List_Remove(vmdata_list, p)) {
					ret = 1;
					break;
				}
				List_Deallocate(vmdata_list, p);
				break;
			}
		}
	} while (!List_Get_Next(vmdata_list, (void **) &p));
	return ret;
}

/*
 * Allocate a vm_data node for storing per VM data.
 */

int LTEXT
vm_allocate(DWORD vm) {
	struct vm_data *p;
	void **pp;

	if (cb_offset == 0 || Test_Sys_VM_Handle(vm)) {
		return 0;
	}

	pp = (void **) (vm + cb_offset);
	p = *pp;

	if (p == NULL) {
		if (vmdata_list == NULL
		    || List_Allocate(vmdata_list, (void **) &p)) {
			*pp = NULL;
			return 0;
		}

		lognum("vm_allocate", vm);
		lognum("vm_data", (DWORD) p);

		memset(p, 0, sizeof *p);

		p->vm = vm;
		p->locked_v86 = 0;
		p->forced_jemm_pt = 0;
		p->using_jemm_pt = 0;
		p->using_my_stack = 0;
		p->assigned_dma_page = 0;
		p->dma_flipflop = 0;
		p->restore_cbfn = 0;
		p->jemm_pt = NULL;
		p->jemm_pt_phys = NULL;
		p->dma_buffer_page = 0;
		p->sblive_fix = 0;
		p->mu_pages = NULL;
		p->fh_pages = NULL;

		List_Attach(vmdata_list, p);
		*pp = p;

		vm_allocate_count++;
		lognum("vm_alloc", vm_allocate_count);

	}
	return 0;
}

/*
 * Deallocate resources used by the VM.
 */
int LTEXT
vm_deallocate(DWORD vm) {
	if (cb_offset != 0) {
		struct vm_data *p = get_vm_data(vm);
			
		if (p != NULL && vmdata_list != NULL) {
			if (vmdata_destroy_node(p, 1)) {
				return 1;
			}
			vm_allocate_count--;
			lognum("vm_dealloc", vm_allocate_count);
		}
		*(void **)(vm + cb_offset) = NULL;
	}
	return 0;
}

/*
 * Allocate resources used globly by the VxD driver.
 */

int LTEXT
device_allocate(void) {
	// DWORD vm, curvm;

	break_page = 
		_PageAllocate(1, PG_SYS, 0,
			      0, 0, 0x100000,
			      &break_page_phys,
			      PAGECONTIG | PAGEUSEALIGN | PAGEFIXED);
	lognum("break_page", (DWORD) break_page);

	if (break_page == NULL) {
		return 1;
	}

	dword_memset((DWORD *) break_page, 0xB90FB90F, PAGE_SIZE / 4);

	if (List_Create(LF_Alloc_Error | LF_Use_Heap, sizeof(struct vm_data), 
			&vmdata_list)) {
		vmdata_list = NULL;
		return 1;
	}

	cb_offset = _Allocate_Device_CB_Area(sizeof(struct vm_data *), 0);
	if (cb_offset == 0) {
		return 1;
	}

	return hook_double_fault();
}

/*
 * Free all the global resources.
 */
int LTEXT
device_deallocate(int pagesafe) {
	lognum("device_deallocate", pagesafe);
	restore_vmm_idt(Get_Cur_VM_Handle());
	if (hooked_task_switch) {
		Cancel_Call_When_Task_Switched(
			TaskSwitchedCallback_Thunk(task_switch));
		hooked_task_switch = 0;
	}
#ifdef USE_LOGRAM
	if (logram_event != NULL) {
		Cancel_Restricted_Event(logram_event);
		logram_event = NULL;
	}
#endif
	if (pagesafe && vmdata_list != NULL) {
		vmdata_destroy_node(NULL, pagesafe);
		List_Destroy(vmdata_list);
		vmdata_list = NULL;
	}
	if (break_page != NULL) {
		_PageFree(break_page, 0);
		break_page = NULL;
	}

	if (pagesafe && cb_offset != 0) {
		_Deallocate_Device_CB_Area(cb_offset, 0);
		cb_offset = 0;
	}

	unhook_double_fault();

	return 0;
}

/*
 * Handle a DeviceIoControl() call from a Win32 programme.
 */

DWORD LTEXT
handle_ioctl(DWORD UNUSED vm, struct DIOCParams *p) {
	switch(p->dwIoControlCode) {
	case DIOC_GETVERSION:
	case VJEMM_GETVERSION:
		lognote("GETVERSION");
		if (p->lpvOutBuffer != NULL && p->cbOutBuffer >= 4) {
			p->lpvOutBuffer[0] = VJEMM_DDB.DDB_Dev_Minor_Version;
			p->lpvOutBuffer[1] = VJEMM_DDB.DDB_Dev_Major_Version;
			p->lpvOutBuffer[2] = VJEMM_API_VERSION & 0xFF;
			p->lpvOutBuffer[3] = (VJEMM_API_VERSION >> 8) & 0xFF;
			*p->lpcbBytesReturned = 4;
		} else {
			*p->lpcbBytesReturned = 0;
		}
		return 0;
		
#ifdef USE_LOG
	case VJEMM_GETLOG:
		lognote("GETLOG");
		return log_read(p->lpvOutBuffer, p->cbOutBuffer,
				p->lpcbBytesReturned);
#endif

	case DIOC_CLOSEHANDLE:
		return 0;
	}

	return ERROR_INVALID_CATEGORY;
}

/*
 * Handle VxD control messages.
 */
int LTEXT
control_proc(DWORD msg, DWORD vm, DWORD esi, 
	     DWORD UNUSED edi, DWORD UNUSED edx) {
	struct vm_data *vmdata;
	// struct cb const *cb;

	switch(msg) {
	case SYS_DYNAMIC_DEVICE_INIT:
#ifdef USE_LOGRAM
		use_logram();
#endif
		lognote("DYN_INIT");
		if (device_allocate()) {
			device_deallocate(1);
			return 1;
		}
		hook_task_switch();
		install_my_idt(Get_Cur_VM_Handle());
		return 0;

	case SYS_DYNAMIC_DEVICE_EXIT:
		lognote("DYN_EXIT");
		lognum("vm_allocated", vm_allocate_count);
#if 0
		if (vm_allocate_count > 0) {
			return 1;
		}
#endif
		return device_deallocate(1);

	case Sys_VM_Terminate:
		lognote("SYS_VM_Terminate");
		device_deallocate(1);
		return 0;

	case System_Exit:
	case Sys_Critical_Exit:
		device_deallocate(0);
		return 0;

#ifdef USE_LOG		
	case Debug_Query:
		lognote("Debug_Query");
		vmdata = get_vm_data(vm);
		Trace_Out_6("#EBX  vmdata: #EAX vmlist: #ECX proplib: #DX"
			    " vmalloc: #ESI nocache: #DI",
			    (DWORD) vmdata, vm, (DWORD) vmdata_list,
			    0, vm_allocate_count,
			    0);
		return 0;
#endif

	case VM_Init:
		lognum("VM_Init", vm);
		return 0;

	case VM_Terminate:
		lognum("VM_Terminate", vm);
		return vm_deallocate(vm);

	case Destroy_VM:
	case VM_Not_Executeable:
		if (msg == Destroy_VM) {
			lognum("Destroy_VM", vm);
		} else {
			lognum("VM_Not_Executable", vm);
		}
		return vm_deallocate(vm);

	case Begin_PM_App:
		lognote("Begin_PM_App");
		fix_pm_idt();
		return 0;

#if 0
		SHELL_SYSMODAL_Message(vm,
				       MB_ASAP | MB_SYSTEMMODAL | MB_NOWINDOW,
				       "Protected mode application",
				       "MyVEJMM");
		return 1;
#endif

	case End_PM_App:
		lognote("End_PM_App");
		return 0;

#if 0
	case Thread_Init:
		lognum("Thread_Init", edi);
		lognum("vm", Get_Cur_VM_Handle());
		break;
#endif

	case Set_Device_Focus:
		lognum("Set_Device_Focus", vm);
		lognum("VID", edx);
		break;

	case W32_DEVICEIOCONTROL:
		return handle_ioctl(vm, (struct DIOCParams *) esi);
	}

	return 0;
}

/*
 * Handle API calls from the V86 mode front end.
 */
void LTEXT
v86_api_proc(DWORD vm, struct Client_Reg_Struc *regs) {
	struct vm_data *vmdata;

	switch(regs->Client_AX) {
	case VJEMM_GETVERSION:
		regs->Client_EAX = VJEMM_DDB.DDB_Dev_Minor_Version;
		regs->Client_AH = VJEMM_DDB.DDB_Dev_Major_Version;
		regs->Client_ECX = VJEMM_API_VERSION;
		regs->Client_DX = VJEMM_VERSION_BUILD;
		CLIENT_CLEAR_CARRY(regs);
		return;

	case VJEMM_FORCE_JEMM_PT:
#ifdef USE_LOG
		log_clear();
#endif
		lognote("FORCE_JEMM_PT");
		vmdata = alloc_vm_data(vm);
		
		if (vmdata != NULL) {
			regs->Client_EAX =
				force_jemm_pt(vmdata, (struct init_packet *)
					      Client_Ptr_Flat(Client_CX,
							      Client_BX));
		}
		CLIENT_CLEAR_CARRY(regs);
		return;

	case VJEMM_RESTORE_PT:
		lognote("RESTORE_PT");
		vmdata = get_vm_data(vm);
		if (vmdata != NULL) {
			unforce_jemm_pt(vmdata);
			vm_deallocate(vm);
		}	
		CLIENT_CLEAR_CARRY(regs);
		return;
	}
	CLIENT_SET_CARRY(regs);
}

Define_Control_Proc_Thunk(control_proc);
Define_API_Proc_Thunk(v86_api_proc);

Declare_Virtual_Device(VJEMM,
		       0, 1,
		       Callback_Thunk_Name(control_proc),
		       UNDEFINED_DEVICE_ID, UNDEFINED_INIT_ORDER,
		       Callback_Thunk_Name(v86_api_proc),
		       Callback_Thunk_Name(v86_api_proc), NULL);

